<?php

namespace Modules\Blog\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Modules\Blog\Entities\Blog;
use Modules\Blog\Entities\BlogCategory;
use Modules\Blog\Entities\Comment;
use Modules\Blog\Http\Requests\CreateComment;
use Modules\Blog\Http\Requests\DeleteBlog;
use Modules\Blog\Http\Requests\DeleteComment;
use Modules\Slider\Entities\Slider;
use Yajra\DataTables\DataTables;
use Yajra\DataTables\Html\Builder;
use Nwidart\Modules\Facades\Module;
use Modules\Core\Http\Controllers\BaseController;

class BlogController extends BaseController
{
    protected $htmlBuilder;
    public function __construct(Builder $htmlBuilder)
    {
        $this->htmlBuilder = $htmlBuilder;
        $this->htmlBuilder->parameters([
            'paging' => true,
            'searching' => true,
            'info' => false,
            'searchDelay' => 350,
            'language' => [
                'url' => Module::asset('core:datatable.json')
            ],
        ]);
        Parent::__construct();
    }
    public function index(Request $request)
    {
        if($request->ajax()) {
            return DataTables::of(Blog::with('user','category')
                ->select(['blogs.id', 'blogs.title','display', 'blogs.created_at', 'blogs.updated_at'
                 ,DB::raw('concat(users.fname ," ",users.lname) as users_fullname ')
                 ,DB::raw('blog_categories.title as blog_categories_title ')
                ])->join('users', 'blogs.user_id', '=', 'users.id')
                ->join('blog_categories', 'blogs.blog_category_id', '=', 'blog_categories.id'))
                ->editColumn('display', function($blog) {

                    if($blog->display==1){
                        return "<span class='btn btn-success'> فعال</span>";
                    }elseif($blog->display==0){
                        return "<span class='btn btn-danger'>غیرفعال</span>";
                    }
                })
                ->addColumn('operation', function($blog) {
                    return parent::redirectAction('list/delete?id='.$blog->id,'حذف','danger mr-1')
                        .parent::redirectAction('list/edit?id='.$blog->id,'ویرایش','warning mr-1');
                })
                ->rawColumns(['operation','display'])
                ->make(true);
        }
        $object = 'بلاگ ها';
        $html = $this->htmlBuilder
            ->addColumn(['data' =>'id', 'name' => 'id', 'title' => 'شناسه'])
            ->addColumn(['data' => 'users_fullname', 'name' => 'users.fname', 'title' => 'نام کاربر'])
            ->addColumn(['data' => 'users_fullname', 'name' => 'users.lname', 'title' => 'نام کاربر','class'=>'d-none'])
            ->addColumn(['data' => 'blog_categories_title', 'name' => 'blog_categories.title', 'title' => 'دسته بندی بلاگ'])
            ->addColumn(['data' => 'title', 'name' => 'title', 'title' => 'عنوان'])
            ->addColumn(['data' => 'display', 'name' => 'display', 'title' => 'نمایش'])
            ->addColumn(['data' => 'created_at', 'name' => 'created_at', 'title' => 'ثبت'])
            ->addColumn(['data' => 'updated_at', 'name' => 'updated_at', 'title' => 'ویرایش'])
            ->addColumn(['data' => 'operation', 'name' => 'operation', 'title' => 'عملیات']);
        return view('core::shared.datatable', compact(['html','object']));
    }

    /**
     * Show the specified resource.
     * @param Request $request
     * @return Response
     * @SWG\Post(
     *      path="/blog/show",
     *      operationId="getBlogById",
     *      tags={"Projects"},
     *      summary="Get blog information",
     *      description="return blog items",
     *      @SWG\Parameter(
     *          name="id",
     *          description="blog id",
     *          required=true,
     *          type="integer",
     *          in="query"
     *      ),
     *      @SWG\Response(
     *          response=200,
     *          description="successful operation"
     *       ),
     *      @SWG\Response(response=400, description="Bad request"),
     *      @SWG\Response(response=404, description="Resource Not Found"),
     *      security={
     *         {
     *             "oauth2_security_example": {"write:projects", "read:projects"}
     *         }
     *     },
     * )
     */
    public function show(Request $request)
    {
        $data=Blog::find($request->id)->with('category','user','comments')->where('id',$request->id)->first();
        $data->increment('hit');
        return response()->json([
            'data'  => $data,
            'message'=>'ok',
            'status' => 200
        ]);
    }
    /**
     * Show the specified resource.
     * @param Request $request
     * @return Response
     */
    public function list(Request $request)
    {
        if($request->id){
            $list=BlogCategory::find($request->id)->blog()->with('category')->orderBy('id','DESC')->get();
        }else{
            $list=Blog::with('category')->orderBy('id','DESC')->get();
        }
        if($request->sort == 'hit'){
            $list=$list->sortByDesc('hit')->values()->all();
        }
        if($request->sort == 'like'){
            $list=$list->sortByDesc('likes')->values()->all();
        }
        $list=BaseController::paginateCollection($list,9,$request->page);
        return response()->json([
            'data'  => $list,
            'message'=>'ok',
            'status' => 200
        ]);
    }
    /**
     * Remove the specified resource from storage.
     * @param DeleteBlog $request
     * @return Response
     */
    public function destroy(DeleteBlog $request)
    {
        Blog::find($request->id)->delete();
        return redirect('admin/blog/list');
    }
    /**
     * This function use to create comment for blog
     * @param CreateComment $request
     * @return Response
     */
    function comment(CreateComment $request){
        $blog=Blog::find($request->id);

            $comment=$blog->comments()->create([
                'comment'=>$request->comment,
                'user_id'=>Auth::user()->id,
            ]);

        $results=Comment::where('id',$comment->id)->with('commentable')->first();
        return response()->json([
            'data'  => $results,
            'message'=>__('ok'),
            'status' => 200
        ]);
    }
    /**
     * This function use to get list og blog comments
     * @param Request $request
     * @return Response
     */
    function commentList(Request $request){
        $data= Blog::where('id',$request->id)->with('comments')->first();
        return response()->json([
            'data'  => $data,
            'message'=>__('ok'),
            'status' => 200
        ]);
    }
    /**
     * Remove the specified comment
     * @param DeleteComment $request
     * @return Response
     */
    public function deleteComment(DeleteComment $request)
    {
        Comment::find($request->id)->delete();
        return response()->json([
            'data' =>true,
            'message'=>'deleted',
            'status' => 200
        ]);
    }
    /**
     * Show the form for creating a new resource.
     * @return Response
     */
    public function add()
    {
        $data['model'] = new Blog();
        $data['route_action'] = 'blog.list.store';
        return view('core::shared.add',$data);
    }
    /**
     * Store a newly created resource in storage.
     * @param  Request $request
     * @return Response
     */
    public function store(Request $request)
    {
        $blog=Blog::create([
            'user_id' => Auth::user()->id,
            'title' => $request->title,
            'seo_title' => $request->seo_title,
            'seo_description' => $request->seo_description,
            'content' => $request->get('content'),
            'description' =>$request->description,
            'blog_category_id' =>$request->blog_category_id,
            'display' =>$request->display
        ]);
        Parent::attachmentProcess($request,$blog);
        return redirect()->route('admin.blog.list');
    }

    /**
     * Show the form for editing the specified resource.
     * @param Request $request
     * @return Response
     */
    public function edit(Request $request)
    {
        $data['route_action'] = 'blog.list.update';
        $data['data']=Blog::find($request->id);
        return view('core::shared.edit', $data);
    }
    /**
     * Update the specified resource in storage.
     * @param  Request $request
     * @return Response
     */
    public function update(Request $request)
    {
        Blog::find(($request->id))->update($request->all());
        $data=Blog::find($request->id);
        Parent::attachmentProcess($request,$data);
        return redirect()->route('admin.blog.list');
    }
    /**
     * this function used to like a blog
     * @param Request $request
     * @return \Illuminate\Contracts\Routing\ResponseFactory|\Symfony\Component\HttpFoundation\Response
     */
    function like(Request $request){
        $blog = Blog::find($request->id);
        $like = $blog->like()->firstOrCreate([
            'user_id'=>Auth::user()->id
        ]);
        if($like->wasRecentlyCreated){
            $blog->increment('like');
        }else{
            $like->delete();
            $blog->decrement('like');
        }
        return response()->json([
            'data'  => true,
            'message'=>'ok',
            'status' => 200
        ]);
    }

    function getLikedBlog(Request $request){
        $blogs=Auth::user()->likes()->with('likable')->get();
        return response()->json([
            'data'  => $blogs,
            'message'=>'ok',
            'status' => 200
        ]);
    }
    /**
     * this function used to search Blog
     * @param Request $request
     * @return array
     */
    function search(Request $request){
        $input = $request->q;

        $data['blog']=Blog::where('title','like',"%$input%")
            ->orWhere('description','like',"%$input%")
            ->orWhere('content','like',"%$input%")
            ->get();

        $data['category']=BlogCategory::where('title','like',"%$input%")
            ->orWhere('description','like',"%$input%")
            ->get();

        return response()->json([
            'data'  => $data,
            'message'=>'ok',
            'status' => 200
        ]);
    }

    static public function display(){
        return ['1'=>'فعال','0' => 'غیر فعال'];
    }

    static public function category(){
        $category= BlogCategory::pluck('title','id');
        return $category;
    }

    public function showAll()
    {
        $data['blog']=Blog::orderBy('id','desc')->with('category')->paginate(9);
        $data['category']=BlogCategory::orderBy('id','desc')->get();
        $data['slider']=Slider::where('slug','main')->with('subs')->get();
        return response()->json([
            'data'  => $data,
            'message'=>'ok',
            'status' => 200
        ]);
    }

}
